// Copyleft 2006 Chris Korda
// This program is free software; you can redistribute it and/or modify it
// under the terms of the GNU General Public License as published by the Free
// Software Foundation; either version 2 of the License, or any later version.
/*
        chris korda

		revision history:
		rev		date	comments
        00      03feb06	initial version

        Video Compression dialog

*/

// VideoComprDlg.cpp : implementation file
//

#include "stdafx.h"
#include "Resource.h"
#include "VideoComprDlg.h"

// these should be members of the VfwCompressDialogs enum
#define VfwCompressDialog_QueryConfig	0x04
#define VfwCompressDialog_QueryAbout	0x08

#define SAFE_RELEASE(p) { if (p) { (p)->Release(); (p) = NULL; } }

#ifdef _DEBUG
#define new DEBUG_NEW
#undef THIS_FILE
static char THIS_FILE[] = __FILE__;
#endif

/////////////////////////////////////////////////////////////////////////////
// CVideoComprDlg dialog

IMPLEMENT_DYNAMIC(CVideoComprDlg, CDialog);

CVideoComprDlg::CVideoComprDlg(CWnd* pParent)
	: CDialog(CVideoComprDlg::IDD, pParent)
{
	//{{AFX_DATA_INIT(CVideoComprDlg)
	m_KeyFrameEdit = 0;
	m_PFramesEdit = 0;
	m_DataRateEdit = 0;
	m_WndSizeEdit = 0;
	//}}AFX_DATA_INIT
	m_hr = 0;
	m_SelCom = 0;
	m_IsCoInit = FALSE;
	m_pGraph = NULL;
	m_pSource = NULL;
	m_pDest = NULL;
	m_pSourceOut = NULL;
	m_pDestIn = NULL;
	m_pComprIn = NULL;
	m_pComprOut = NULL;
}

CVideoComprDlg::~CVideoComprDlg()
{
	DisconnectCompr();	// disconnect compressor and remove it from graph
	SAFE_RELEASE(m_pGraph);	// release graph reference if any
	// free all resources held by compressor info list
	for (int i = 0; i < m_Name.GetSize(); i++) {
		INFO	*ip = &m_Info[i];
		SAFE_RELEASE(ip->pVidCom);
		SAFE_RELEASE(ip->pVfwDlg);
		SAFE_RELEASE(ip->pCompr);
		SAFE_RELEASE(ip->pMoniker);
		delete [] ip->pDlgState;
		ip->pDlgState = NULL;
	}
	if (m_IsCoInit)	// if COM was initialized
		CoUninitialize();	// uninitialize it
}

void CVideoComprDlg::DoDataExchange(CDataExchange* pDX)
{
	CDialog::DoDataExchange(pDX);
	//{{AFX_DATA_MAP(CVideoComprDlg)
	DDX_Control(pDX, IDC_VCD_QUAL_NUM, m_QualNum);
	DDX_Control(pDX, IDC_VCD_QUALITY, m_Quality);
	DDX_Control(pDX, IDC_VCD_COMBO, m_Combo);
	DDX_Text(pDX, IDC_VCD_KEY_FRAME_EDIT, m_KeyFrameEdit);
	DDX_Text(pDX, IDC_VCD_P_FRAMES_EDIT, m_PFramesEdit);
	DDX_Text(pDX, IDC_VCD_DATA_RATE_EDIT, m_DataRateEdit);
	DDX_Text(pDX, IDC_VCD_WND_SIZE_EDIT, m_WndSizeEdit);
	//}}AFX_DATA_MAP
}

bool CVideoComprDlg::SetFilters(IBaseFilter *pSource, IBaseFilter *pDest)
{
	if (pSource == NULL || pDest == NULL)	// validate arguments
		return FALSE;
	SAFE_RELEASE(m_pGraph);	// release previous graph reference if any
	FILTER_INFO	Info;
	// assume both filters are currently in the same graph
	m_hr = pSource->QueryFilterInfo(&Info);	// get source filter info
	if (FAILED(m_hr) || Info.pGraph == NULL) {
		AfxMessageBox(IDS_VCDE_GET_GRAPH);
		return FALSE;
	}
	m_pGraph = Info.pGraph;	// QueryFilterInfo added a reference to graph
	m_pSource = pSource;
	m_pDest = pDest;
	return TRUE;
}

bool CVideoComprDlg::ConnectCompr()
{
	if (m_pGraph == NULL)	// SetFilters wasn't called, or didn't succeed
		return FALSE;
	INFO	*ip = GetCurInfo();
	if (ip->pCompr == NULL)	// compressor hasn't been created
		return FALSE;
	m_hr = m_pGraph->AddFilter(ip->pCompr, L"Compressor");	// add compressor to graph
	if (SUCCEEDED(m_hr)) {
		// get pins and connect source -> compressor -> destination
		m_pSourceOut	= CDirShowU::GetPin(m_pSource, PINDIR_OUTPUT);
		m_pDestIn		= CDirShowU::GetPin(m_pDest, PINDIR_INPUT);
		m_pComprIn		= CDirShowU::GetPin(ip->pCompr, PINDIR_INPUT);
		m_pComprOut		= CDirShowU::GetPin(ip->pCompr, PINDIR_OUTPUT);
		m_hr = m_pGraph->ConnectDirect(m_pSourceOut, m_pComprIn, NULL);
		if (SUCCEEDED(m_hr))
			m_hr = m_pGraph->ConnectDirect(m_pComprOut, m_pDestIn, NULL);
	}
	return SUCCEEDED(m_hr);
}

void CVideoComprDlg::DisconnectCompr()
{
	if (m_pGraph != NULL) {
		m_pGraph->Disconnect(m_pSourceOut);	// disconnect all pins
		m_pGraph->Disconnect(m_pDestIn);
		m_pGraph->Disconnect(m_pComprIn);
		m_pGraph->Disconnect(m_pComprOut);
		m_pGraph->RemoveFilter(GetCompr());	// remove compressor from graph
	}
	SAFE_RELEASE(m_pSourceOut);	// release pins
	SAFE_RELEASE(m_pDestIn);
	SAFE_RELEASE(m_pComprIn);
	SAFE_RELEASE(m_pComprOut);
}

bool CVideoComprDlg::SelectCompr()
{
	DisconnectCompr();	// disconnect compressor and remove it from graph
	int sel = m_Combo.GetCurSel();	// get current selection from combo box
	if (sel < 0)	// no selection
		return FALSE;
	m_SelCom = m_Combo.GetItemData(sel);	// map item position to compressor index
	if (m_SelCom < 0)	// combo box error
		return FALSE;
	INFO	*ip = GetCurInfo();
	if (ip->pCompr == NULL) {	// if compressor isn't already created
		IMoniker *pMoniker = ip->pMoniker;	// get compressor's moniker
		if (pMoniker == NULL)
			return FALSE;
		m_hr = pMoniker->BindToObject(NULL, NULL,	// create compressor
			IID_IBaseFilter, (void **)&ip->pCompr);
		if (FAILED(m_hr))
			return FALSE;
		m_hr = ip->pCompr->QueryInterface(	// get IAMVfwCompressDialogs
			IID_IAMVfwCompressDialogs, (void **)&ip->pVfwDlg);
		if (SUCCEEDED(m_hr)) {	// query dialog support
			ip->CanConfig = ip->pVfwDlg->ShowDialog(	// Configure dialog
				VfwCompressDialog_QueryConfig, 0) == S_OK;
			ip->CanAbout = ip->pVfwDlg->ShowDialog(		// About dialog
				VfwCompressDialog_QueryAbout, 0) == S_OK;
		}
		IPin	*pComprOut = CDirShowU::GetPin(ip->pCompr, PINDIR_OUTPUT);
		if (pComprOut != NULL) {	// if we got compressor's output pin
			m_hr = pComprOut->QueryInterface(	// get IAMVideoCompression
				IID_IAMVideoCompression, (void **)&ip->pVidCom);
			if (SUCCEEDED(m_hr)) {	// retrieve compressor's defaults and capabilities
				int cbVersion;
				int cbDescrip;
				m_hr = ip->pVidCom->GetInfo(NULL, &cbVersion, NULL, &cbDescrip, 
					&ip->KeyFrameRate, &ip->PFramesPerKey, &ip->Quality, &ip->Caps);
				if (SUCCEEDED(m_hr)) {
					ip->CanQuality	= (ip->Caps & CompressionCaps_CanQuality) != 0;
					ip->CanCrunch	= (ip->Caps & CompressionCaps_CanCrunch) != 0;
					ip->CanKeyFrame = (ip->Caps & CompressionCaps_CanKeyFrame) != 0;
					ip->CanBFrame	= (ip->Caps & CompressionCaps_CanBFrame) != 0;
					ip->CanWindow	= (ip->Caps & CompressionCaps_CanWindow) != 0;
				}
			}
			pComprOut->Release();
		}
	}
	// compressor must be connected, else Vfw Configure dialog ignores SetState
	ip->CanConnect = ConnectCompr();
	return TRUE;
}

void CVideoComprDlg::UpdateQuality()
{
	INFO	*ip = GetCurInfo();
	int qual = ip->CanQuality ? int(ip->Quality * 100.0 + .5) : 0;
	m_Quality.SetPos(qual);
	CString s;
	s.Format("%d", qual);
	m_QualNum.SetWindowText(s);
}

void CVideoComprDlg::UpdateUI()
{
	INFO	*ip = GetCurInfo();
	// update quality
	GetDlgItem(IDC_VCD_CONFIG)->EnableWindow(ip->CanConfig);
	GetDlgItem(IDC_VCD_ABOUT)->EnableWindow(ip->CanAbout);
	GetDlgItem(IDC_VCD_QUAL_CAP)->EnableWindow(ip->CanQuality);
	m_Quality.EnableWindow(ip->CanQuality);
	m_QualNum.EnableWindow(ip->CanQuality);
	// update data rate
	GetDlgItem(IDC_VCD_DATA_RATE_EDIT)->EnableWindow(ip->CanCrunch);
	GetDlgItem(IDC_VCD_DATA_RATE_CAP)->EnableWindow(ip->CanCrunch);
	GetDlgItem(IDC_VCD_DATA_RATE_UNIT)->EnableWindow(ip->CanCrunch);
	m_DataRateEdit = ip->CanCrunch ? ip->DataRate / 1000 : 0;
	// update key frame rate
	GetDlgItem(IDC_VCD_KEY_FRAME_EDIT)->EnableWindow(ip->CanKeyFrame);
	GetDlgItem(IDC_VCD_KEY_FRAME_CAP)->EnableWindow(ip->CanKeyFrame);
	GetDlgItem(IDC_VCD_KEY_FRAME_UNIT)->EnableWindow(ip->CanKeyFrame);
	m_KeyFrameEdit = ip->CanKeyFrame ? ip->KeyFrameRate : 0;
	// update P frames per key
	GetDlgItem(IDC_VCD_P_FRAMES_EDIT)->EnableWindow(ip->CanBFrame);
	GetDlgItem(IDC_VCD_P_FRAMES_CAP)->EnableWindow(ip->CanBFrame);
	GetDlgItem(IDC_VCD_P_FRAMES_UNIT)->EnableWindow(ip->CanBFrame);
	m_PFramesEdit = ip->CanBFrame ? ip->PFramesPerKey : 0;
	// update window size
	GetDlgItem(IDC_VCD_WND_SIZE_EDIT)->EnableWindow(ip->CanWindow);
	GetDlgItem(IDC_VCD_WND_SIZE_CAP)->EnableWindow(ip->CanWindow);
	GetDlgItem(IDC_VCD_WND_SIZE_UNIT)->EnableWindow(ip->CanWindow);
	m_WndSizeEdit = ip->CanWindow ? ip->WindowSize : 0;
	// update dialog from member data
	UpdateData(FALSE);
	UpdateQuality();
}

BEGIN_MESSAGE_MAP(CVideoComprDlg, CDialog)
	//{{AFX_MSG_MAP(CVideoComprDlg)
	ON_BN_CLICKED(IDC_VCD_ABOUT, OnAbout)
	ON_BN_CLICKED(IDC_VCD_CONFIG, OnConfig)
	ON_CBN_SELCHANGE(IDC_VCD_COMBO, OnSelchangeCombo)
	ON_WM_DESTROY()
	ON_WM_HSCROLL()
	ON_EN_KILLFOCUS(IDC_VCD_KEY_FRAME_EDIT, OnKillfocusKeyFrameEdit)
	ON_EN_KILLFOCUS(IDC_VCD_P_FRAMES_EDIT, OnKillfocusPFramesEdit)
	ON_EN_KILLFOCUS(IDC_VCD_DATA_RATE_EDIT, OnKillfocusDataRateEdit)
	ON_EN_KILLFOCUS(IDC_VCD_WND_SIZE_EDIT, OnKillfocusWndSizeEdit)
	//}}AFX_MSG_MAP
END_MESSAGE_MAP()

/////////////////////////////////////////////////////////////////////////////
// CVideoComprDlg message handlers

BOOL CVideoComprDlg::OnInitDialog() 
{
	CDialog::OnInitDialog();
	
	HRESULT	m_hr = CoInitializeEx(NULL, COINIT_APARTMENTTHREADED);
	if (FAILED(m_hr)) {
		AfxMessageBox(IDS_VCDE_INIT_COM);
		return TRUE;
	}
	m_IsCoInit = TRUE;
	// enumerate video compressors
	CPtrArray	Moniker;
	CDirShowU::EnumDevs(CLSID_VideoCompressorCategory, m_Name, Moniker);
	// add uncompressed video to end of compressor list
	CString uc((LPCSTR)IDS_VCD_UNCOMPRESSED);
	m_Name.Add(uc);
	Moniker.Add(NULL);
	// propagate combo box and info array
	INFO	Info;
	ZeroMemory(&Info, sizeof(Info));
	int devs = m_Name.GetSize();
	int	pos = -1;	// if sorted combo, item position and compressor index differ
	for (int i = 0; i < devs; i++) {		// for each compressor
		pos = m_Combo.AddString(m_Name[i]);		// add its name to combo box
		m_Combo.SetItemData(pos, i);			// store its index in item data
		Info.pMoniker = (IMoniker *)Moniker[i];	// copy its moniker to info struct
		m_Info.Add(Info);						// add info struct to info array
	}
	m_SelCom = devs - 1;	// default selection is uncompressed video
	m_Combo.SetCurSel(pos);	// set default selection in combo box
	UpdateUI();

	return TRUE;  // return TRUE unless you set the focus to a control
				  // EXCEPTION: OCX Property Pages should return FALSE
}

void CVideoComprDlg::OnDestroy() 
{
	CDialog::OnDestroy();
	DisconnectCompr();	// make compressor available to user
}

void CVideoComprDlg::OnOK() 
{
	INFO	*ip = GetCurInfo();
	if (ip->pCompr != NULL && !ip->CanConnect) {	// if compressor can't connect
		CString	Err((LPCSTR)IDS_VCDE_CONNECT), DSErr;
		CDirShowU::GetErrorString(m_hr, DSErr);
		AfxMessageBox(Err + "\n" + DSErr);	// display error message
	} else
		CDialog::OnOK();
}

void CVideoComprDlg::OnConfig() 
{
	IAMVfwCompressDialogs	*pVfwDlg = GetCurInfo()->pVfwDlg;
	if (pVfwDlg != NULL) {
		INFO	*ip = GetCurInfo();
		if (ip->pDlgState != NULL) {	// if dialog state was saved
			m_hr = pVfwDlg->SetState(ip->pDlgState, ip->szDlgState);	// restore it
			if (FAILED(m_hr))
				AfxMessageBox(IDS_VCDE_SET_STATE);
		}
		m_hr = pVfwDlg->ShowDialog(VfwCompressDialog_Config, m_hWnd);	// show dialog
		delete [] ip->pDlgState;	// delete previous state
		ip->pDlgState = NULL;	// mark it deleted
		if (SUCCEEDED(m_hr)) {
			int sz;
			m_hr = pVfwDlg->GetState(NULL, &sz);	// get size of dialog state
			if (SUCCEEDED(m_hr)) {
				ip->pDlgState = new char[sz];
				ip->szDlgState = sz;
				m_hr = pVfwDlg->GetState(ip->pDlgState, &sz);	// save dialog state
				if (FAILED(m_hr))
					AfxMessageBox(IDS_VCDE_SET_STATE);
			} else
				AfxMessageBox(IDS_VCDE_SET_STATE);
		} else
			AfxMessageBox(IDS_VCDE_SHOW_CONFIG);
	}
}

void CVideoComprDlg::OnAbout() 
{
	IAMVfwCompressDialogs	*pVfwDlg = GetCurInfo()->pVfwDlg;
	if (pVfwDlg != NULL) {
		m_hr = pVfwDlg->ShowDialog(VfwCompressDialog_About, m_hWnd);	// show dialog
		if (FAILED(m_hr))
			AfxMessageBox(IDS_VCDE_SHOW_ABOUT);
	}
}

void CVideoComprDlg::OnSelchangeCombo() 
{
	SelectCompr();
	UpdateUI();
}

void CVideoComprDlg::OnHScroll(UINT nSBCode, UINT nPos, CScrollBar* pScrollBar) 
{
	INFO	*ip = GetCurInfo();
	if (ip->CanQuality) {
		ip->Quality = m_Quality.GetPos() / 100.0;
		UpdateQuality();
		if (ip->pVidCom != NULL) {
			m_hr = ip->pVidCom->put_Quality(ip->Quality);
			if (FAILED(m_hr))
				AfxMessageBox(IDS_VCDE_SET_QUALITY);
		}
	}
	CDialog::OnHScroll(nSBCode, nPos, pScrollBar);
}

void CVideoComprDlg::OnKillfocusDataRateEdit() 
{
	UpdateData(TRUE);	// retrieve member data from dialog
	INFO	*ip = GetCurInfo();
	ip->DataRate = m_DataRateEdit * 1000;	// convert from Kbps to bps
	VIDEOINFOHEADER	vih;
	m_hr = CDirShowU::RWVideoInfo(ip->pCompr, vih, FALSE);	// read video info
	if (SUCCEEDED(m_hr)) {
		vih.dwBitRate = ip->DataRate;
		m_hr = CDirShowU::RWVideoInfo(ip->pCompr, vih, TRUE);	// write video info
	}
	if (FAILED(m_hr))
		AfxMessageBox(IDS_VCDE_SET_DATA_RATE);
}

void CVideoComprDlg::OnKillfocusKeyFrameEdit() 
{
	UpdateData(TRUE);	// retrieve member data from dialog
	INFO	*ip = GetCurInfo();
	if (ip->pVidCom != NULL) {
		ip->KeyFrameRate = m_KeyFrameEdit;
		m_hr = ip->pVidCom->put_KeyFrameRate(ip->KeyFrameRate);
		if (FAILED(m_hr))
			AfxMessageBox(IDS_VCDE_SET_KEY_FRAMES);
	}
}

void CVideoComprDlg::OnKillfocusPFramesEdit() 
{
	UpdateData(TRUE);	// retrieve member data from dialog
	INFO	*ip = GetCurInfo();
	if (ip->pVidCom != NULL) {
		ip->PFramesPerKey = m_PFramesEdit;
		m_hr = ip->pVidCom->put_PFramesPerKeyFrame(ip->PFramesPerKey);
		if (FAILED(m_hr))
			AfxMessageBox(IDS_VCDE_SET_PFRAMES);
	}
}

void CVideoComprDlg::OnKillfocusWndSizeEdit() 
{
	UpdateData(TRUE);	// retrieve member data from dialog
	INFO	*ip = GetCurInfo();
	if (ip->pVidCom != NULL) {
		ip->WindowSize = m_WndSizeEdit;
		m_hr = ip->pVidCom->put_WindowSize(ip->WindowSize);
		if (FAILED(m_hr))
			AfxMessageBox(IDS_VCDE_SET_WND_SIZE);
	}
}
